/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _PDE_H_
#define _PDE_H_

/*==================================================================================================
 Include Files
==================================================================================================*/
#include <stdint.h>
#include "EmbeddedTypes.h"
#include "fsl_common.h"
#include "pde_driver.h"

/*!
 * @addtogroup GENFSK_Localization
 * @addtogroup PhaseBaseDistEstimate Phase-based Distance Estimation localization feature
 * @ingroup GENFSK_Localization
 * Accurate distance measurement.
 * See "Localization Quick Start Guide" document for explanation on how to operate with PDE API.
 * @{
 */

/*! @file
 * GENFSK Localization - Phase-based Distance Estimation high level service
 */

/*==================================================================================================
 User definitions, might be redefined in app_preinclude.h
==================================================================================================*/
#ifndef PDE_USE_MEM_MANAGER
#define PDE_USE_MEM_MANAGER             (1)
#endif

/*! Select which memory manager to use for PDE: 1 = SDK memory manager, 2 = malloc/free */
#ifndef PDE_USE_FLIB
#define PDE_USE_FLIB                    (1)
#endif

/* Debug options */
#define PDE_DEBUG_ADD_CARRIER_OFFSET    (0)
#define PDE_DC_OFFSET_REMOVAL           (0)

#ifndef PDE_MAX_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ
#define PDE_MAX_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ	(16U)
#endif

#ifndef PDE_MIN_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ
#define PDE_MIN_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ	(4U)
#endif

#ifndef PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ
#define PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ     (16U)
#endif

#if (PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ > PDE_MAX_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ) || \
    (PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ < PDE_MIN_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ)
#error PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ is incorrect
#endif

/*==================================================================================================
 Type definitions
==================================================================================================*/

/*!
 * PDE completion event.
 * Event returned by PDE callback to notify about a PDE event.
 * @see pde_event_callback_t
 */
typedef enum _pde_event_tag {
    kPdeEvent_CaptureComplete_c,
    kPdeEvent_ErrorOccurred_c,
}pde_event_t;

/*!
 * PDE error code.
 * Code returned by PDE callback to notify about a PDE error.
 * @see pde_event_callback_t
 */
typedef enum _pde_error_tag{
    kPdeError_Ok_c,
    kPdeError_SamplesBufferAlreadyFlushed_c,
    kPdeError_DmaCaptureFailure_c,
    kPdeError_Timeout_c,
    kPdeError_CrcInvalid_c,
    kPdeError_GfsxRxError_c,
    kPdeError_MaxFrequencyIndexReached_c,
    kPdeError_MaxRetriesReached_c,
    kPdeError_SnMismatch_c,
    kPdeError_PhaseInvalid_c,
    kPdeError_StartPacketTimeout_c,
    kPdeError_CalibrationPacketTimeout_c,
    kPdeError_SyncPacketTimeout_c,
    kPdeError_DataSharingTimeout_c,
    kPdeError_DqiFilterDiscarded_c,
    kPdeError_UnknownOpCode_c,
}pde_error_t;

/*!
 * Possible returned status from API.
 */
typedef enum _pde_status_tag{
    kPdeStatus_Ok_c,                      /*!< success */
    kPdeStatus_InvalidInputParams_c,      /*!< Given parameters are invalid */
    kPdeStatus_NotReady_c,                /*!< Module is not ready to process this request */
    kPdeStatus_InternalError_c,           /*!< A software Internal error occured */
    kPdeStatus_OutOfMemory_c,             /*!< A memory allocation failure error occured */
}pde_status_t;

/*!
 * Structure to report measurement settings.
 * @see PDE_GetMeasurementSettings
 */
typedef struct _pde_measure_settings_tag{
    uint32_t samplesCaptured;
    uint8_t agcIndex;
}pde_measure_settings_t;

/*!
 * Application callback to catch PDE events.
 * Invoked when a measurement is complete.
 * @param event The type of notification raised by PDE API
 * @param error Possible error code
 *
 * @see pde_event_t, pde_error_t, PDE_StartDataCapture
 */
typedef void (*pde_event_callback_t) (pde_event_t event, pde_error_t error);

/*! @brief PDE report types. */
#define PDE_REPORT_RAW_IQ             (1U<<0U) /*!< Reports captured IQ samples (RD&MD) */
#define PDE_REPORT_PHASES             (1U<<1U) /*!< Reports computed phases (RD&MD)     */

/*!
 * Description of parameters used during PDE data capture
 * @see PDE_StartDataCapture
 */
typedef struct pde_capture_config_tag
{
    uint16_t  reportFlags;                    /*!< Flags defining the type of data to be captured and reported */
} pde_capture_config_t;

/*!
 * Description of parameters used during PDE system initialisation
 * @see PDE_Init
 */
typedef struct _pde_system_config_tag
{
    pde_event_callback_t pdeEventCallback;   /*!< The application callback to be invoked by PDE when an event occurs */
    uint32_t localAccessAddress;             /*!< Access Address used for localization */
    uint32_t peerAccessAddress;              /*!< Access Address of peer */
    uint32_t freqSweepStartHz;               /*!< First frequency to use during freq sweep in Hz */
    uint32_t freqSweepIndentHz;              /*!< Frequency increment for various frequencies used during freq sweep */
    uint16_t freqSweepNb;                    /*!< Number of frequencies to use during freq sweep */
    uint8_t  nbOfSamplesPerFrequency;        /*!< Number of samples to capture per frequency. Supported values are 32, 16, 8 and 4 */
} pde_system_config_t;

/*==================================================================================================
 Public prototypes
==================================================================================================*/

/*!
 * Initialize system config structure to default values.
 * This initializer must be called before assigning values.
 */

void PDE_InitSysConfig(pde_system_config_t *pdeSysCfg);

/*!
 * Initialize capture config structure to default values.
 * This initializer must be called before assigning values.
 */
void PDE_InitCaptureConfig(pde_capture_config_t *pdeCaptureCfg);

/*!
 * Set-up the device for PDE usage.
 * This function must be called before using this module and after each call to PDE_Stop.
 * @param [in] pdeSysCfg Pointer to configuration parameters to initialize PDE system
 * @return status
 * @see PDE_InitSysConfig
 */
pde_status_t PDE_Init (pde_system_config_t *pdeSysCfg);

/*!
 * Stop any PDE activity.
 * Restore TPM timer.
 */
void PDE_Stop(void);

/*!
 * Start listening for PDE start request.
 * The device will listen for start request forever. Once a measurement is complete,
 * it will start listening again.
 */
void PDE_StartListening(void);

/*!
 * Start PDE measurement.
 * Upon completion, preprocessing of IQ samples will be called. Application is notified
 * thanks to the callback, and results can be obtained from various PDE_FlushXXX functions.
 * There is one and only one callback invocation per call to PDE_StartDataCapture.
 * @param [in] pCaptureCfg Pointer to configuration parameters for the requested capture
 * @param [out] pOutputPhaseBuffer Buffer provided by the application to store resulting phases
 * @param [out] pFreqMask Buffer provided by the application to store valid frequency mask (each bit represents frequency status)
 * @see pde_event_callback_t
 * @return status
 */
pde_status_t PDE_StartDataCapture (pde_capture_config_t *pCaptureCfg, int16_t *pOutputPhaseBuffer, uint32_t *pFreqMask);

/*!
 * Set options used during phase preprocessing.
 * @param [in] frequencyMaskEnable If enabled, frequency might be flagged invalid in case of saturation
 * @param [in] phaseMaskEnable If enabled, frequency might be flagged invalid if frequency max/min phase are outside 0.4 rads range
 */
void PDE_SetPreprocessingOptions(bool_t frequencyMaskEnable, bool_t phaseMaskEnable);

/*!
 * Flush IQ samples collected on the local device.
 * Samples correspond to last measurement.
 * Arrays must be sized to samplesCaptured as returned by PDE_GetMeasurementSettings.
 * @param [out] iSamples Buffer provided by the application to store I samples
 * @param [out] qSamples Buffer provided by the application to store Q samples
 */
pde_error_t PDE_FlushLocalIqSamples (int16_t* iSamples, int16_t* qSamples);

/*!
 * Flush IQ samples collected on the peer device.
 * Samples correspond to last measurement.
 * Arrays must be sized to samplesCaptured as returned by PDE_GetMeasurementSettings.
 * @param [out] iSamples Buffer provided by the application to store I samples
 * @param [out] qSamples Buffer provided by the application to store Q samples
 */
pde_error_t PDE_FlushPeerIqSamples (int16_t* iSamples, int16_t* qSamples);

/*!
 * Calculate phase for each IQ sample provided.
 * @param [in] iqSampleNumber Number of samples provided
 * @param [in] iSamples I samples (fixed point S3.12)
 * @param [in] qSamples Q samples (fixed point S3.12)
 * @param [out] outputBuffer Buffer provided by the application to store phase per sample (fixed point S3.12)
 */
pde_error_t PDE_CalculateIqPhase (uint16_t iqSampleNumber, int16_t* iSamples, int16_t* qSamples, int16_t* outputBuffer);

/*!
 * Flush phase for samples collected on the local device.
 * One phase per frequency is provided. Array must be sized to PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES.
 * @param [out] pOutputBuffer Buffer provided by the application to store phases
 */
void PDE_FlushLocalPreprocessedPhase (int16_t *pOutputBuffer);

/*!
 * Flush phase for samples collected on the peer device.
 * One phase per frequency is provided. Array must be sized to PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES.
 * @param [out] pOutputBuffer Buffer provided by the application to store phases
 */
void PDE_FlushPeerPreprocessedPhase (int16_t *pOutputBuffer);

/*!
 * Get Measurement settings.
 * @param [out] settings Buffer provided by the application to store settings used during last measurement.
 */
void PDE_GetMeasurementSettings (pde_measure_settings_t *settings);

/*!
 * Obtain pointers to PLL settings as used during last measurement.
 * @param [out] PllSettings PLL Settings has 3 elements of 32-bit each for every frequency
 * @param [out] CoraseTune Coarse Tune is 1 element of 8-bit for every frequency
 * @param [out] NumberFrequencies number of frequencies (size of arrays above)
 */
void PDE_GetPllSettings (pde_freq_calibration_t** PllSettings, uint8_t** CoarseTune, uint16_t * NumberFrequencies);

/*! @} */

#endif //_PDE_H_
